import chokidar from 'chokidar';
import fs from 'fs'
import fse from 'fs-extra';
import child_process from 'child_process';

// ====================================================================
// |  
// ====================================================================
function log(message) {
  const options = { 
    encoding: "utf-8", 
    flag: "a+", 
  };
  const datetime = new Date().toISOString()
  message = `[${datetime}] ${message}\n`
  fse.outputFileSync('/logs/builder/assets.log', message, options);
  console.log(message)
}

// ====================================================================
// |    
// ====================================================================
function copyAllIntoWWW(paths) {
  return new Promise((resolve, reject) => {
    for(const path of paths) {
      log(path)
      if (!fs.existsSync(path)) {
        log(`Directory does not exist in project: ${path}`)
        continue;
      }

      var publicPath = path
      publicPath = publicPath.replace('/app','/www')
      try {
        child_process.execSync(`mkdir -p ${publicPath} & cp -rf ${path}/* ${publicPath}`);
        log(`Done copying: ${path}`)
      } catch (err) {
        console.error(err)
      }
    }
  });
}

// ====================================================================
// |    
// ====================================================================
function copyIntoWWW(path) {

  // Do not compile inside .delete folders
  if (path.includes(".delete")) {
    return
  }

  log(path);

  var publicPath = path
  publicPath = publicPath.replace('/app','/www')

  try {
    child_process.execSync(`cp -rf ${path} ${publicPath}`);
    log(`Done copying: ${path}`)
  } catch (err) {
    console.error(err)
  }
}

// ====================================================================
// |    
// ====================================================================
function deleteFromWWW(path) {

  // Do not compile inside .delete folders
  if (path.includes(".delete")) {
    return
  }

  log(path);

  var publicPath = path
  publicPath = publicPath.replace('/app','/www')

  try {
    child_process.execSync(`rm -rf ${path} ${publicPath}`);
    log(`Done deleting: ${path}`)
  } catch (err) {
    console.error(err)
  }
}


// ====================================================================
// |    
// ====================================================================
function startWatcher(watchFolders) {
  // https://github.com/paulmillr/chokidar
  const options = {
    awaitFinish: true,
    ignoreInitial: true,
    usePolling: true,
    interval: 100,    
    depth: 5
  }

  const watcher = chokidar.watch(watchFolders, options).on('all', (event, path) => {
    log(event, path);
  });

  watcher
    .on('add', path => copyIntoWWW(path))
    .on('addDir', path => copyIntoWWW(path))
    .on('change', path => copyIntoWWW(path))
    .on('unlink', path => deleteFromWWW(path))
    .on('unlinkDir', path => deleteFromWWW(path));

  watcher
    .on('ready', () => log('**********    Watch is running    **********'));
}


// ====================================================================
// |    MAIN
// ====================================================================
async function go() {
  const watchFolders = [
    '/app/css',
    '/app/docs',
    '/app/fonts',
    '/app/images',
    '/app/libs',
    '/app/lib'
  ]  
  if (process.argv[2] == 'recompile') {
    await copyAllIntoWWW(watchFolders)
    startWatcher(watchFolders);
  } else {
    startWatcher(watchFolders);
  }
}

go()