import chokidar from 'chokidar';
import fs from 'fs'
import fse from 'fs-extra';
import * as sass from 'sass';
import fg from 'fast-glob';

// ====================================================================
// |  
// ====================================================================
function log(message) {
  const options = { 
    encoding: "utf-8", 
    flag: "a+", 
  };
  const datetime = new Date().toISOString()
  message = `[${datetime}] ${message}\n`
  fse.outputFileSync('/logs/builder/sass.log', message, options);
  console.log(message)
}

// ====================================================================
// |     
// ====================================================================
function compile(path) {
  log('-----------------------');

  // Do not compile inside .delete folders
  if (path.includes(".delete")) {
    return
  }

  const filename = path.split("/").slice(-1).toString()
  if (filename.startsWith("_")) {
    compileAll()
    return
  }

  const options = {
    logger: {
      warn(message, options) {
        if (options.span) {
          log('====================================================================================================================')
          log(`Warning when compiling: ${options.span.url.pathname}:`)
          log(`  ${message}\n`)
        }
      },
      debug(message, options) {
        if (options.span) {
          log('====================================================================================================================')
          log(`Debug message compiling: ${options.span.url.pathname}:`)
          log(`  ${message}\n`)
        }
      }        
    }
  }

  try {
    const result = sass.compile(path, options);
    var publicPath = path
    publicPath = publicPath.replace('/app/scss','/www/css')
    publicPath = publicPath.replace('.scss','.css')
    fse.outputFileSync(publicPath, result.css);
  } catch (err) {
    log(`[ERROR] SASS Compile: ${path}`)
    log(err)
  }

}

// ====================================================================
// |    
// ====================================================================
function compileParents(path) {

  // Do not compile inside .delete folders
  if (path.includes(".delete")) {
    return
  }
    
  // Do not compile partials (begins with _)
  const filename = path.split("/").slice(-1).toString()
  if (filename.startsWith("_")) {
    return
  }

  // Do not compile if the file does not end with .scss
  if (!path.endsWith(".scss")) {
    return
  }    

  const options = {
    logger: {
      warn(message, options) {
        if (options.span) {
          log('=============================')
          log(`Warning when compiling: ${options.span.url.pathname}:`)
          log(`  ${message}\n`)
        }
      },
      debug(message, options) {
        if (options.span) {
          log('=============================')
          log(`Debug message compiling: ${options.span.url.pathname}:`)
          log(`  ${message}\n`)
        }
      }        
    }
  }

  try {
    const result = sass.compile(path, options);
    var publicPath = path
    publicPath = publicPath.replace('/app/scss','/www/css')
    publicPath = publicPath.replace('.scss','.css')
    fse.outputFileSync(publicPath, result.css);
  } catch (err) {
    log(`[ERROR] SASS Compile: ${path}`)
    log(err)
  }
  

}

// ====================================================================
// |    
// ====================================================================
function compileAll(glob) {
  return new Promise((resolve, reject) => {
    const sassFiles = fg.globSync(glob)
    for (const file of sassFiles) {
      compileParents(file);
    }
    log(`Done`)
  });
}

// ====================================================================
// |    
// ====================================================================
function startWatcher(watchFolders) {

  // https://github.com/paulmillr/chokidar
  const options = {
    awaitFinish: true,
    ignoreInitial: true,
    usePolling: true,
    interval: 100    
  }

  const watcher = chokidar.watch(watchFolders, options).on('all', (event, path) => {
    log(`${path} was ${event}`);
  });

  watcher
    .on('add', path => compile(path))
    .on('change', path => compile(path))
    .on('unlink', path => compile(path));

  watcher
    .on('ready', () => log('**********    Watch is running    **********'));

}

// ====================================================================
// |    MAIN
// ====================================================================
async function go() {
  const watchFolders = [
    '/app/scss/**/*.scss'
  ]
  if (process.argv[2] == 'recompile') {
    await compileAll('/app/scss/**/*.scss')
    startWatcher(watchFolders);
  } else {
    startWatcher(watchFolders);
  }
}

go()